#include "winxy.h"
#include "optimize_performance.h"
#include <utility>
#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QPushButton>
#include <QLabel>
#include <QProcess>
#include <QDesktopServices>
#include <QUrl>
#include <QFileInfo>
#include <QPainter>
#include <QDir>
#include <QPixmap>
#include <QCoreApplication>
#include <QFileInfoList>

OptimizePerformanceWidget::OptimizePerformanceWidget(QWidget *parent)
    : QWidget(parent)
{
    setupUI();
}

// --- Helper Functions ---

// 1. Generic Asset Loader
QPixmap loadAsset(const QString &fileName) {
    // Use the centralized AppConfig to locate assets reliably
    // This uses the broadcasted root directory found during startup
    QString path = AppConfig::getAssetPath(fileName);

    if (QFileInfo::exists(path)) {
        QPixmap pix(path);
        if (!pix.isNull()) return pix;
    }
    return QPixmap();
}

// 2. Fallback Icon Generator
QPixmap generateFallbackIcon(int size, const QString &text, const QColor &bgColor) {
    QPixmap pixmap(size, size);
    pixmap.fill(bgColor);
    QPainter painter(&pixmap);
    painter.setRenderHint(QPainter::Antialiasing);

    int padX = size * 0.08;
    int padY = size * 0.25;
    painter.setBrush(Qt::black);
    painter.setPen(Qt::NoPen);
    painter.drawRoundedRect(padX, padY, size - (padX * 2), size - (padY + (size * 0.12)), 6, 6);

    painter.setPen(Qt::white);
    int fontSize = (text.length() > 10) ? size * 0.10 : size * 0.12;
    QFont font("Segoe UI", fontSize, QFont::Bold);
    painter.setFont(font);
    painter.drawText(pixmap.rect(), Qt::AlignCenter, text);
    painter.end();
    return pixmap;
}

// 3. Smart Path Finder
QString findAppPath(const QString &defaultPath, const QString &folderName, const QString &exeName) {
    // 1. Check the hardcoded default path first (Fastest)
    QString nativeDefault = QDir::toNativeSeparators(defaultPath);
    if (QFileInfo::exists(nativeDefault)) {
        return nativeDefault;
    }

    // 2. Lightweight Search: Iterate over all logical drives
    QFileInfoList drives = QDir::drives();

    // Common directories where software is usually installed
    QStringList commonDirs;
    commonDirs << "Program Files"
               << "Program Files (x86)"
               << "Games"
               << "Software"
               << ""; // Check root of drive (Handles D:\FurMark_win64 cases)

    for (const QFileInfo &drive : std::as_const(drives)) {
        QString driveRoot = drive.absoluteFilePath();

        for (const QString &dir : std::as_const(commonDirs)) {
            // Construct candidate path
            QString candidate = driveRoot + dir + "/" + folderName + "/" + exeName;

            // Clean up separators (handle double slashes)
            QString cleanPath = QDir::toNativeSeparators(QDir::cleanPath(candidate));

            if (QFileInfo::exists(cleanPath)) {
                return cleanPath;
            }
        }
    }

    // 3. Not found
    return QString();
}

// --- Main UI Setup ---

void OptimizePerformanceWidget::setupUI()
{
    QVBoxLayout *layout = new QVBoxLayout(this);
    layout->setContentsMargins(40, 40, 40, 40);
    layout->setSpacing(10);
    layout->setAlignment(Qt::AlignTop);

    // --- Title ---
    QLabel *title = new QLabel("System Optimization", this);
    title->setStyleSheet("font-size: 24px; font-weight: 600; color: #333; border: none; margin-bottom: 5px;");
    layout->addWidget(title);

    // --- Description ---
    QLabel *desc = new QLabel("Manage power plans and system settings for gaming/work.", this);
    desc->setStyleSheet("font-size: 14px; color: #666; border: none; margin-bottom: 10px;");
    layout->addWidget(desc);

    // --- Styles ---
    QString featureDescStyle = "font-size: 13px; color: #555; margin-top: 10px; margin-bottom: 2px;";
    QString warnStyle = "color: #C0392B;";
    QString noteStyle = "color: #0078d4;";
    QString btnStyle =
        "QPushButton { "
        "  background-color: #ffffff; "
        "  color: #333; "
        "  border: 1px solid #ccc; "
        "  border-radius: 6px; "
        "  padding: 12px; "
        "  font-size: 15px; "
        "  text-align: left; "
        "}"
        "QPushButton:hover { border: 1px solid #0078d4; background-color: #f0f8ff; }"
        "QPushButton:pressed { background-color: #e0e0e0; }";

    // ==========================================
    // Existing Optimization Buttons
    // ==========================================

    // 1. Power Options
    QLabel *lblPowerDesc = new QLabel(this);
    lblPowerDesc->setTextFormat(Qt::RichText);
    lblPowerDesc->setText("Adjust power plans (Balanced/High Performance).<br>"
                          "<span style='" + warnStyle + "'>High performance modes usually consume more power and generate more heat.</span>");
    lblPowerDesc->setStyleSheet(featureDescStyle);
    layout->addWidget(lblPowerDesc);

    QPushButton *btnPower = new QPushButton("🔋 Configure Power Options", this);
    btnPower->setCursor(Qt::PointingHandCursor);
    btnPower->setStyleSheet(btnStyle);
    connect(btnPower, &QPushButton::clicked, this, [=](){
        QProcess::startDetached("control", QStringList() << "powercfg.cpl");
    });
    layout->addWidget(btnPower);

    // 2. Screen & Sleep
    QLabel *lblSleepDesc = new QLabel(this);
    lblSleepDesc->setTextFormat(Qt::RichText);
    lblSleepDesc->setText("Manage screen timeout, sleep timers, and Energy Saver mode.<br>"
                          "<span style='" + warnStyle + "'>Disabling sleep completely may degrade battery life and display hardware.</span>");
    lblSleepDesc->setStyleSheet(featureDescStyle);
    layout->addWidget(lblSleepDesc);

    QPushButton *btnSleep = new QPushButton("💤 Screen, Sleep and Energy Saver", this);
    btnSleep->setCursor(Qt::PointingHandCursor);
    btnSleep->setStyleSheet(btnStyle);
    connect(btnSleep, &QPushButton::clicked, this, [=](){
        QProcess::startDetached("explorer", QStringList() << "ms-settings:powersleep");
    });
    layout->addWidget(btnSleep);

    // 3. Game Mode
    QLabel *lblGameDesc = new QLabel(this);
    lblGameDesc->setTextFormat(Qt::RichText);
    lblGameDesc->setText("Optimize Windows to prioritize gaming processes.<br>"
                         "<span style='" + warnStyle + "'>May minimize background activity, potentially affecting non-game apps.</span>");
    lblGameDesc->setStyleSheet(featureDescStyle);
    layout->addWidget(lblGameDesc);

    QPushButton *btnGameMode = new QPushButton("🎮 Game Mode Settings", this);
    btnGameMode->setCursor(Qt::PointingHandCursor);
    btnGameMode->setStyleSheet(btnStyle);
    connect(btnGameMode, &QPushButton::clicked, this, [=](){
        QProcess::startDetached("explorer", QStringList() << "ms-settings:gaming-gamemode");
    });
    layout->addWidget(btnGameMode);

    // 4. Memory Integrity
    QLabel *lblCoreDesc = new QLabel(this);
    lblCoreDesc->setTextFormat(Qt::RichText);
    lblCoreDesc->setText("Toggle Core Isolation (Memory Integrity) virtualization security.<br>"
                         "<span style='" + warnStyle + "'>Disabling reduces security; Enabling might be incompatible with some older drivers.</span>");
    lblCoreDesc->setStyleSheet(featureDescStyle);
    layout->addWidget(lblCoreDesc);

    QPushButton *btnCoreIso = new QPushButton("🛡️ Manage Memory Integrity", this);
    btnCoreIso->setCursor(Qt::PointingHandCursor);
    btnCoreIso->setStyleSheet(btnStyle);
    connect(btnCoreIso, &QPushButton::clicked, this, [=](){
        QProcess::startDetached("explorer", QStringList() << "windowsdefender://coreisolation");
    });
    layout->addWidget(btnCoreIso);

    // 5. Advanced Graphics
    QLabel *lblGraphDesc = new QLabel(this);
    lblGraphDesc->setTextFormat(Qt::RichText);
    lblGraphDesc->setText("Manage Auto HDR, Windowed Optimizations, GPU Scheduling, and VRR.<br>"
                          "<span style='" + noteStyle + "'>Features depend on hardware support. Restart games after changing.</span>");
    lblGraphDesc->setStyleSheet(featureDescStyle);
    layout->addWidget(lblGraphDesc);

    QPushButton *btnGraphics = new QPushButton("🖥️ Graphical Settings (Windows)", this);
    btnGraphics->setCursor(Qt::PointingHandCursor);
    btnGraphics->setStyleSheet(btnStyle);
    connect(btnGraphics, &QPushButton::clicked, this, [=](){
        QProcess::startDetached("explorer", QStringList() << "ms-settings:display-advancedgraphics");
    });
    layout->addWidget(btnGraphics);

    // ==========================================
    // NEW SECTION: Recommended Software
    // ==========================================
    layout->addSpacing(30);

    QLabel *swTitle = new QLabel("Recommended Software", this);
    swTitle->setStyleSheet("font-size: 20px; font-weight: 600; color: #333; border: none; margin-bottom: 2px;");
    layout->addWidget(swTitle);

    QLabel *swHint = new QLabel("Click icon to open the software or official website.", this);
    swHint->setStyleSheet("font-size: 11px; color: #888; font-style: italic; border: none; margin-bottom: 10px;");
    layout->addWidget(swHint);

    // --- SOFTWARE 1: NVIDIA APP ---
    QWidget *swContainer1 = new QWidget(this);
    swContainer1->setStyleSheet("background-color: #f9f9f9; border: 1px solid #e0e0e0; border-radius: 8px;");
    QHBoxLayout *swLayout1 = new QHBoxLayout(swContainer1);
    swLayout1->setContentsMargins(15, 15, 15, 15);
    swLayout1->setSpacing(15);

    QPushButton *btnNvidia = new QPushButton(this);
    btnNvidia->setFixedSize(96, 96);
    btnNvidia->setCursor(Qt::PointingHandCursor);
    btnNvidia->setStyleSheet("QPushButton { border: none; background: transparent; border-radius: 8px; } QPushButton:hover { background-color: #e0e0e0; }");

    QPixmap nvPixmap = loadAsset("nvidia.png");
    if (!nvPixmap.isNull()) btnNvidia->setIcon(QIcon(nvPixmap.scaled(96, 96, Qt::IgnoreAspectRatio, Qt::SmoothTransformation)));
    else btnNvidia->setIcon(QIcon(generateFallbackIcon(96, "NVIDIA\nAPP", QColor("#76b900"))));
    btnNvidia->setIconSize(QSize(96, 96));

    QVBoxLayout *nvTextLayout = new QVBoxLayout();
    nvTextLayout->setSpacing(5);
    QLabel *lblNvTitle = new QLabel("NVIDIA App", this);
    lblNvTitle->setStyleSheet("font-size: 16px; font-weight: bold; color: #76b900; border: none;");
    QLabel *lblNvDesc = new QLabel("The modern companion for PC gamers. Use it to keep Game Ready Drivers updated, configure G-SYNC and Color settings, enable Auto GPU Tuning, and optimize game performance with a unified interface.", this);
    lblNvDesc->setStyleSheet("font-size: 12px; color: #555; border: none;");
    lblNvDesc->setWordWrap(true);
    lblNvDesc->setAlignment(Qt::AlignTop | Qt::AlignLeft);
    nvTextLayout->addWidget(lblNvTitle);
    nvTextLayout->addWidget(lblNvDesc);
    nvTextLayout->addStretch();
    swLayout1->addWidget(btnNvidia);
    swLayout1->addLayout(nvTextLayout);
    swLayout1->setStretch(1, 1);
    layout->addWidget(swContainer1);

    connect(btnNvidia, &QPushButton::clicked, this, [=](){
        QString nvPath = "C:/Program Files/NVIDIA Corporation/NVIDIA app/CEF/NVIDIA App.exe";
        QString nativePath = QDir::toNativeSeparators(nvPath);
        if (QFileInfo::exists(nativePath)) QProcess::startDetached("explorer.exe", QStringList() << nativePath);
        else QDesktopServices::openUrl(QUrl("https://www.nvidia.com/en-eu/software/nvidia-app/"));
    });

    layout->addSpacing(10);

    // --- SOFTWARE 2: MSI AFTERBURNER ---
    QWidget *swContainer2 = new QWidget(this);
    swContainer2->setStyleSheet("background-color: #f9f9f9; border: 1px solid #e0e0e0; border-radius: 8px;");
    QHBoxLayout *swLayout2 = new QHBoxLayout(swContainer2);
    swLayout2->setContentsMargins(15, 15, 15, 15);
    swLayout2->setSpacing(15);

    QPushButton *btnMsi = new QPushButton(this);
    btnMsi->setFixedSize(96, 96);
    btnMsi->setCursor(Qt::PointingHandCursor);
    btnMsi->setStyleSheet("QPushButton { border: none; background: transparent; border-radius: 8px; } QPushButton:hover { background-color: #e0e0e0; }");

    QPixmap msiPixmap = loadAsset("msi_afterburner.png");
    if (!msiPixmap.isNull()) btnMsi->setIcon(QIcon(msiPixmap.scaled(96, 96, Qt::IgnoreAspectRatio, Qt::SmoothTransformation)));
    else btnMsi->setIcon(QIcon(generateFallbackIcon(96, "MSI\nAFTERBURNER", QColor("#CC0000"))));
    btnMsi->setIconSize(QSize(96, 96));

    QVBoxLayout *msiTextLayout = new QVBoxLayout();
    msiTextLayout->setSpacing(5);
    QLabel *lblMsiTitle = new QLabel("MSI Afterburner", this);
    lblMsiTitle->setStyleSheet("font-size: 16px; font-weight: bold; color: #CC0000; border: none;");
    QLabel *lblMsiDesc = new QLabel("The most recognized and widely used graphics card overclocking utility. It provides detailed hardware monitoring, fan profile control, and allows you to push your GPU performance to the limit (Manual GPU Overclocking).", this);
    lblMsiDesc->setStyleSheet("font-size: 12px; color: #555; border: none;");
    lblMsiDesc->setWordWrap(true);
    lblMsiDesc->setAlignment(Qt::AlignTop | Qt::AlignLeft);
    msiTextLayout->addWidget(lblMsiTitle);
    msiTextLayout->addWidget(lblMsiDesc);
    msiTextLayout->addStretch();
    swLayout2->addWidget(btnMsi);
    swLayout2->addLayout(msiTextLayout);
    swLayout2->setStretch(1, 1);
    layout->addWidget(swContainer2);

    connect(btnMsi, &QPushButton::clicked, this, [=](){
        QString foundPath = findAppPath("C:/Program Files (x86)/MSI Afterburner/MSIAfterburner.exe", "MSI Afterburner", "MSIAfterburner.exe");
        if (!foundPath.isEmpty()) QProcess::startDetached("explorer.exe", QStringList() << foundPath);
        else QDesktopServices::openUrl(QUrl("https://www.msi.com/Landing/afterburner/graphics-cards"));
    });

    layout->addSpacing(10);

    // --- SOFTWARE 3: GPU-Z ---
    QWidget *swContainer3 = new QWidget(this);
    swContainer3->setStyleSheet("background-color: #f9f9f9; border: 1px solid #e0e0e0; border-radius: 8px;");
    QHBoxLayout *swLayout3 = new QHBoxLayout(swContainer3);
    swLayout3->setContentsMargins(15, 15, 15, 15);
    swLayout3->setSpacing(15);

    QPushButton *btnGpuZ = new QPushButton(this);
    btnGpuZ->setFixedSize(96, 96);
    btnGpuZ->setCursor(Qt::PointingHandCursor);
    btnGpuZ->setStyleSheet("QPushButton { border: none; background: transparent; border-radius: 8px; } QPushButton:hover { background-color: #e0e0e0; }");

    QPixmap gpuzPixmap = loadAsset("gpu_z.png");
    if (!gpuzPixmap.isNull()) btnGpuZ->setIcon(QIcon(gpuzPixmap.scaled(96, 96, Qt::IgnoreAspectRatio, Qt::SmoothTransformation)));
    else btnGpuZ->setIcon(QIcon(generateFallbackIcon(96, "GPU-Z", QColor("#0078d4"))));
    btnGpuZ->setIconSize(QSize(96, 96));

    QVBoxLayout *gpuzTextLayout = new QVBoxLayout();
    gpuzTextLayout->setSpacing(5);
    QLabel *lblGpuZTitle = new QLabel("TechPowerUp GPU-Z", this);
    lblGpuZTitle->setStyleSheet("font-size: 16px; font-weight: bold; color: #0078d4; border: none;");
    QLabel *lblGpuZDesc = new QLabel("A lightweight system utility designed to provide vital information about your video card. It displays detailed specifications including GPU model, memory type, clock speeds, sensors, and manufacturing details.", this);
    lblGpuZDesc->setStyleSheet("font-size: 12px; color: #555; border: none;");
    lblGpuZDesc->setWordWrap(true);
    lblGpuZDesc->setAlignment(Qt::AlignTop | Qt::AlignLeft);
    gpuzTextLayout->addWidget(lblGpuZTitle);
    gpuzTextLayout->addWidget(lblGpuZDesc);
    gpuzTextLayout->addStretch();
    swLayout3->addWidget(btnGpuZ);
    swLayout3->addLayout(gpuzTextLayout);
    swLayout3->setStretch(1, 1);
    layout->addWidget(swContainer3);

    connect(btnGpuZ, &QPushButton::clicked, this, [=](){
        QString foundPath = findAppPath("C:/Program Files (x86)/GPU-Z/GPU-Z.exe", "GPU-Z", "GPU-Z.exe");
        if (!foundPath.isEmpty()) QProcess::startDetached("explorer.exe", QStringList() << foundPath);
        else QDesktopServices::openUrl(QUrl("https://www.techpowerup.com/gpuz/"));
    });

    layout->addSpacing(10);

    // --- SOFTWARE 4: GEEKS3D FURMARK ---
    QWidget *swContainer4 = new QWidget(this);
    swContainer4->setStyleSheet("background-color: #f9f9f9; border: 1px solid #e0e0e0; border-radius: 8px;");
    QHBoxLayout *swLayout4 = new QHBoxLayout(swContainer4);
    swLayout4->setContentsMargins(15, 15, 15, 15);
    swLayout4->setSpacing(15);

    QPushButton *btnFurMark = new QPushButton(this);
    btnFurMark->setFixedSize(96, 96);
    btnFurMark->setCursor(Qt::PointingHandCursor);
    btnFurMark->setStyleSheet("QPushButton { border: none; background: transparent; border-radius: 8px; } QPushButton:hover { background-color: #e0e0e0; }");

    QPixmap furPixmap = loadAsset("furmark.jpg");
    if (!furPixmap.isNull()) {
        btnFurMark->setIcon(QIcon(furPixmap.scaled(96, 96, Qt::IgnoreAspectRatio, Qt::SmoothTransformation)));
    } else {
        // Fallback: Orange/Brown for the "Donut"
        btnFurMark->setIcon(QIcon(generateFallbackIcon(96, "FURMARK", QColor("#E65100"))));
    }
    btnFurMark->setIconSize(QSize(96, 96));

    QVBoxLayout *furTextLayout = new QVBoxLayout();
    furTextLayout->setSpacing(5);
    QLabel *lblFurTitle = new QLabel("Geeks3D FurMark", this);
    lblFurTitle->setStyleSheet("font-size: 16px; font-weight: bold; color: #E65100; border: none;"); // Orange Title
    QLabel *lblFurDesc = new QLabel("The standard OpenGL benchmark for GPU stress testing. Famous for its 'donut' rendering, it pushes your graphics card to the limit to test stability, cooling performance, and power consumption under load.", this);
    lblFurDesc->setStyleSheet("font-size: 12px; color: #555; border: none;");
    lblFurDesc->setWordWrap(true);
    lblFurDesc->setAlignment(Qt::AlignTop | Qt::AlignLeft);
    furTextLayout->addWidget(lblFurTitle);
    furTextLayout->addWidget(lblFurDesc);
    furTextLayout->addStretch();
    swLayout4->addWidget(btnFurMark);
    swLayout4->addLayout(furTextLayout);
    swLayout4->setStretch(1, 1);
    layout->addWidget(swContainer4);

    connect(btnFurMark, &QPushButton::clicked, this, [=](){
        // [UPDATED] Use Smart Search for FurMark
        // Default: C:/Program Files/FurMark_win64/FurMark_GUI.exe
        // Folder: FurMark_win64
        // Exe: FurMark_GUI.exe
        // Note: findAppPath checks drive roots (like D:/FurMark_win64/) automatically
        QString foundPath = findAppPath(
            "C:/Program Files/FurMark_win64/FurMark_GUI.exe",
            "FurMark_win64",
            "FurMark_GUI.exe"
            );

        if (!foundPath.isEmpty()) {
            QProcess::startDetached("explorer.exe", QStringList() << foundPath);
        } else {
            QDesktopServices::openUrl(QUrl("https://geeks3d.com/furmark/"));
        }
    });

    layout->addStretch();
}
